/*
 * Copyright European Commission's
 * Taxation and Customs Union Directorate-General (DG TAXUD).
 */
package eu.europa.ec.taxud.cesop.domain;

import static eu.europa.ec.taxud.cesop.domain.ValidationResultTypeEnum.FULLY_REJECTED;
import static eu.europa.ec.taxud.cesop.domain.ValidationResultTypeEnum.PARTIALLY_REJECTED;

/**
 * Enum for the validation error types.
 * The holder is also used in VM and requires the type ID without data access so it is part of the enum.
 */
public enum ValidationErrorTypeEnum {

    MH_BR_0010("10010", "MessageRefID is not unique.", FULLY_REJECTED),
    /**
     * @deprecated Business rule was removed in version 1.6.0.
     */
    @SuppressWarnings("unused") @Deprecated
    MH_BR_0020("10020", "The 'Timestamp' element refers to a wrong value.", FULLY_REJECTED),
    MH_BR_0030("10030", "The period is before 01/01/2024.", FULLY_REJECTED),
    MH_BR_0040("10040", "Wrong CorrMessageRefID.", FULLY_REJECTED),
    MH_BR_0050("10050", "Wrong format of the MessageRefID.", FULLY_REJECTED),
    MH_BR_0060("10060", "Wrong format of the CorrMessageRefID.", FULLY_REJECTED),
    MH_BR_0070("10070", "An initial Payment Data message can only contain new data.", FULLY_REJECTED),
    MH_BR_0080("10080", "A correction message can only contain corrections and/or deletions.", FULLY_REJECTED),
    MH_BR_0090("10090", "The message is not a payment data message.", FULLY_REJECTED),
    MH_BR_0100("10100", "The reporting period cannot be updated.", FULLY_REJECTED),
    MH_BR_0110("10110", "CorrMessageRefId wrongly used in MessageSpec.", FULLY_REJECTED),
    MH_BR_0120("10120", "Wrong TransmittingCountry", FULLY_REJECTED),
    CM_BR_0010("20010", "DocRefID is not unique within the message.", PARTIALLY_REJECTED),
    CM_BR_0020("20020", "DocRefID is not unique within the system.", PARTIALLY_REJECTED),
    CM_BR_0030("20030", "Wrong DocRefID format.", FULLY_REJECTED),
    CM_BR_0040("20040", "Wrong CorrDocRefId value.", PARTIALLY_REJECTED),
    CM_BR_0050("20050", "CorrDocRefId for new data.", PARTIALLY_REJECTED),
    CM_BR_0060("20060", "Missing CorrDocRefId.", PARTIALLY_REJECTED),
    CM_BR_0070("20070", "CorrDocRefId no longer valid.", PARTIALLY_REJECTED),
    CM_BR_0100("20100", "Invalid PSP's BIC format.", FULLY_REJECTED),
    CM_BR_0110("20110", "Missing ReportedPayee.", FULLY_REJECTED),
    CM_BR_0120("20120", "CorrDocRefId does not belong to corrected message.", FULLY_REJECTED),
    CM_BR_0130("20130", "Discrepancy in the provision of an 'other' type and its specification (full rejection).", FULLY_REJECTED),
    CM_BR_0140("20140", "Discrepancy in the provision of an 'other' type and its specification (partial rejection).", PARTIALLY_REJECTED),
    CM_BR_0150("20150", "Same payee reported under two different ReportedPayee elements.", FULLY_REJECTED),
    /**
     * @deprecated Business rule was removed in version 1.6.0.
     */
    @SuppressWarnings("unused") @Deprecated
    PS_BR_0010("30010", "Business identifier of the Representative is equal to the Business Identifier of the Reporting PSP.", PARTIALLY_REJECTED),
    RP_BR_0010("40010", "The reported transaction does not represent cross-border payment.", PARTIALLY_REJECTED),
    RP_BR_0020("40020", "Wrong IBAN format.", PARTIALLY_REJECTED),
    RP_BR_0030("40030", "IBAN is not valid.", PARTIALLY_REJECTED),
    RP_BR_0040("40040", "ReportedPayee listed in the No payment data for the requested period message.", FULLY_REJECTED),
    RP_BR_0050("40050", "The 'ReportedTransaction' element is missing.", PARTIALLY_REJECTED),
    RP_BR_0060("40060", "Discrepancy in the 'AccountIdentifier' attributes.", PARTIALLY_REJECTED),
    RP_BR_0070("40070", "Invalid Representative's BIC format.", PARTIALLY_REJECTED),
    RP_BR_0080("40080", "Discrepancy between 'AccountIdentifier' and 'Representative'.", PARTIALLY_REJECTED),
    RP_BR_0090("40090", "The 'ReportedTransaction' element is present.", PARTIALLY_REJECTED),
    RP_BR_0100("40100", "Invalid account identifier(s) for the same reported payee.", PARTIALLY_REJECTED),
    RP_BR_0110("40110", "IBAN, BIC or OBAN reported in the accountIdentifierOther element.", PARTIALLY_REJECTED),
    RT_BR_0010("45010", "Wrong value of the 'IsRefund' element in the 'AmountCurrency' element.", PARTIALLY_REJECTED),
    RT_BR_0030("45030", "None of the 'DateTime' elements refers to a date within the reporting period.", PARTIALLY_REJECTED),
    RT_BR_0040("45040", "The 'TransactionIdentifier' element is not unique within the Payment data message.", PARTIALLY_REJECTED),
    RT_BR_0050("45050", "The 'TransactionIdentifier' element is not unique within the system.", PARTIALLY_REJECTED),
    RT_BR_0060("45060", "Zero value for 'Amount' element.", PARTIALLY_REJECTED),
    RT_BR_0080("45080", "Same transaction date provided more than once.", PARTIALLY_REJECTED),
    RT_BR_0090("45090", "Wrong value of the 'IsRefund' element while CorrTransactionIdentifier is provided.", PARTIALLY_REJECTED),
    CM_TR_0010("50010", "The XML message is not well formed.", FULLY_REJECTED),
    CM_TR_0020("50020", "Failed Decryption.", FULLY_REJECTED),
    CM_TR_0030("50030", "Failed Decompression.", FULLY_REJECTED),
    CM_TR_0040("50040", "Failed Signature Check.", FULLY_REJECTED),
    CM_TR_0050("50050", "Failed Threat Scan.", FULLY_REJECTED),
    CM_TR_0060("50060", "Failed Virus Scan.", FULLY_REJECTED),
    CM_TR_0070("50070", "Message size exceeded.", FULLY_REJECTED),
    CM_TR_0080("50080", "Too many errors detected.", FULLY_REJECTED),
    CM_TR_9999("99999", "Custom error.", FULLY_REJECTED);

    private final String code;
    private final String description;
    private final ValidationResultTypeEnum resultType;

    ValidationErrorTypeEnum(final String code, final String description, final ValidationResultTypeEnum resultType) {
        this.code = code;
        this.description = description;
        this.resultType = resultType;
    }

    public static boolean isSupported(ValidationErrorTypeEnum errorCode, String xsdVersion) {
        // attempted structured version
        switch (errorCode){
            case MH_BR_0120:
                return Double.parseDouble(xsdVersion) >= 4.01D;
            case CM_BR_0150:
            case RT_BR_0090:
                return Double.parseDouble(xsdVersion) >= 4.03D;
            default:
                return true;
        }
    }

    public String getCode() {
        return this.code;
    }

    public String getDescription() {
        return this.description;
    }

    public ValidationResultTypeEnum getResultType() {
        return this.resultType;
    }
}
